/** @file   soundsource.h
 * @brief   Declaration of SoundSource - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_SOUNDSOURCE_H
#define H_ENG2D_SOUNDSOURCE_H

#include "eng2d_dll.h"
#include "vec2d.h"
#include "sample.h"

namespace eng2d {


/** @class  SoundSource
 * @brief   A sound sources emit sound and can be placed where ever you want.
 * @author  Tomi Lamminsaari
 *
 * The positional sounds can be created with the instances of this SoundSource -
 * class. These sounds can be attached to any Allegro - sample and they
 * can be placed at any coordinate. As long as you keep calling the
 * @c update() - method, you achieve a sound that fades in and out as the
 * distance to the listener changes.
 */
class DLLIMPORT SoundSource : public Sample
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** The maximum distance from listener when the soundsource still can be
   * heard.
   */
  static float maxDistance;

  /** Maximum panning change from center in percents. (0 - 100) */
  static float maxPanSwing;
  
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	
	/** Constructs new sound source at given coordinate. This sound will be
	 * loopped so it will be played as long as you call the @c turnOff() - method.
   * @param     soundname         Name of this sound. This name is usually
   *                              needed to tell which sample should be played.
   * @param     pos               Location of this soundsource
   */
  SoundSource( const std::string& soundname, const Vec2D& pos );
  
  /** Con


	/** Destructor
   */
	virtual ~SoundSource();


private:

	/** Copy constructor.
   * @param     rO                Reference to another SoundSource
   */
  SoundSource( const SoundSource& rO );

	/** Assignment operator
   * @param     rO                Reference to another SoundSource
   * @return    Reference to us.
   */
  SoundSource& operator = ( const SoundSource& rO );

public:


  ///
  /// Methods
  /// =======

  /** Sets the position of this sample.
   * @param     rP                Location
   */
  void position( const Vec2D& rP );
  
  /** Sets the loop on or off
   * @param     l                 Set <code>true</code> to set loop on.
   */
  void loop( bool l );
  
  /** Sets the loudness of this sound source.
   * @param     loud              New loudness-value. This is the volume we
   *                              use when the distance from this soundsource
   *                              is 0.
   */
  void loudness( int loud );
  

  /** Turns this soundsource on. In other words, starts the audio playback.
   */
  void turnOn();
  
  /** Turns this soundsource off.
   */
  void turnOff();
  
  
  /** Updates this soundsource. Modifies the volume and panning of this
   * soundsource to match the case we are listening this sound at given
   * location.
   * @param     rListener         Coordinates of the listener.
   */
  void update( const Vec2D& rListener );
  
  



  ///
  /// Getter methods
  /// ==============
  
  /** Returns the position of this sample.
   * @return    Current position.
   */
  Vec2D position() const;
  
  /** Returns the name of this soundsource.
   * @return    Name of this soundsource
   */
  const std::string& name() const;
  
  /** Tells if this sound source is on.
   * @return    <code>true</code> if this sound source is on.
   */
  bool active() const;

  /** Returns the loudness of this soundsource. This is the maximum
   * volume the sample has when its distance from listener is 0
   * @return    The loudness of this soundsource
   */
  int loudness() const;



protected:

  ///
  /// Protected mehods
  /// ================

  /** Calculates the volume we should use when we are at given distance
   * from listener.
   * @param     dist              Distance from the listener.
   * @return    The volume we should use.
   */
  int calculateVolume( float dist ) const;
  
  /** Calculates the panning value. This method plays with the difference
   * between the x-coordinates of us and the listener.
   * @param     rListener         Position of the listener in pixels.
   * @return    The panning value ranging from 0 to 255 where 0 is left
   *            and 255 is right.
   */
  int calculatePanning( const Vec2D& rListener ) const;



  ///
  /// Members
  /// =======

  /** Position of this soundsource. */
  Vec2D   m_pos;
  
  /** Name of this sound source */
  std::string m_name;
  
  /** The maximum volume */
  int m_loudness;
  
  /** Which voice was allocated for this sound source */
  int m_voiceIndex;
  
  /** The loop-flag */
  bool m_loop;

  /** Set when this sound source is activated. */
  bool m_active;
  
  /** Set, when soundsource is so far away from listener that we don't
   * keep the soundcard's voice allocated for it.
   */
  bool m_voiceReleased;

  /** Timer that calculates how long the sound has been off. */
//  int m_offTimer;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
